<?php
/* 
* Grav CMS 框架
* PluginPath: /user/plugins/sitesearch/sitesearch.php
* 作者：laok
* 该插件实现了搜索站内文章，或相关分类等信息。
*/

namespace Grav\Plugin;

use Grav\Common\Plugin;
use Grav\Common\Page\Collection;
use Grav\Common\Uri;
use Composer\Autoload\ClassLoader;

class SiteSearchPlugin extends Plugin
{
    public static function getSubscribedEvents()
    {
        return [
            'onPagesInitialized' => [
                ['autoload', 100001],
                ['onPagesInitialized', 0],
            ],
            'onTwigTemplatePaths' => ['onTwigTemplatePaths', 0]
        ];
    }

     /**
     * Add current directory to twig lookup paths.
     */
    public function onTwigTemplatePaths()
    {   $path = str_replace('\\', '/', __DIR__ . '/templates');
        $this->grav['twig']->twig_paths[] = $path;
        // $this->grav['twig']->twig_paths[] = __DIR__ . '/templates';
        // 添加主题覆盖路径（可选）
        // $theme = $this->grav['theme'];
        // $this->grav['twig']->twig_paths[] = "user/plugins/loadimages/templates";
    }

    /**
     * [onPluginsInitialized:100000] Composer autoload.
     *
     * @return ClassLoader
     */
    public function autoload()
    {
        return require __DIR__ . '/vendor/autoload.php';
    }

    
    public function onPagesInitialized()
    {
        if ($this->isAdmin()) return;

        $uri = $this->grav['uri'];
        $route = $this->config->get('plugins.sitesearch.route') ?? '/search';

        if ($uri->path() === $route) {
            $this->handleRequest();
        }
    }

    private function handleRequest()
    {
        $this->sitesearch();
    }

    private function sitesearch()
    {
        // 获取配置参数
        $config = $this->config->get('plugins.sitesearch');
        $excludePatterns = $config['exclude_patterns'] ?? ['#/@eaDir(/|$)#', '#/private/#', '#/test/#'];
        $summaryLength = $config['summary_length'] ?? 250;
        $enableHighlight = $config['enable_highlight'] ?? true;
    
        // 获取关键词
        $keywords = isset($_GET['keywords']) ? trim($_GET['keywords']) : '';
        $keywords = preg_split('/[\s\+]+/', $keywords);
    
        // 获取分页参数
        $page = isset($_GET['page']) ? (int) $_GET['page'] : 1;
        // $perPage = isset($_GET['per_page']) ? (int) $_GET['per_page'] : 10;
        $perPage = isset($_GET['per_page']) ? (int) $_GET['per_page'] : ($this->config->get('plugins.sitesearch.per_page') ?? 10);
        if ($page < 1) $page = 1;
        if ($perPage < 1) $perPage = 10;
    
        $matchedResults = [];
        $totalResultsCount = 0;
    
        // 获取所有页面
        $allPages = $this->grav['pages']->all();
    
        // 遍历所有页面，筛选匹配
        foreach ($allPages as $pageItem) {
            if ($this->isExcluded($pageItem->route(), $excludePatterns)) {
                continue;
            }
    
            $content = $this->getPageContent($pageItem);
            $data = [
                'title' => $pageItem->title(),
                'url' => $pageItem->url(),
                'content' => $content,
                'taxonomy' => $pageItem->taxonomy(),
            ];
    
            // 判断关键词
            if ($this->matchKeywords($keywords, $data)) {
                $matchedResults[] = [
                    'title' => $this->highlight($data['title'], $keywords, $enableHighlight),
                    'link' => $data['url'],
                    'content' => $this->getSummary($content, $keywords, $summaryLength, $enableHighlight),
                    'category' => $data['taxonomy']['category'] ?? [],
                    'tag' => $data['taxonomy']['tag'] ?? [],
                ];
            }
        }
    
        // 当前匹配到的总结果数
        $totalResultsCount = count($matchedResults);
    
        // 排序（根据需要，可启用）
        // 例如按日期排序
        // usort($matchedResults, function($a, $b) {
        //     return strtotime($b['date']) - strtotime($a['date']);
        // });
    
        // 计算分页偏移
        $offset = ($page - 1) * $perPage;
        $resultsPage = array_slice($matchedResults, $offset, $perPage);
    
        // 返回数据（包含分页信息）
        header('Content-Type: application/json');
        echo json_encode([
            'results' => $resultsPage,
            'total' => $totalResultsCount,
            'per_page' => $perPage,
            'page' => $page,
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit();
    }

    // 辅助方法：检查是否在排除目录
    private function isExcluded($route, $patterns)
    {
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $route)) {
                return true;
            }
        }
        return false;
    }

    // 辅助方法：获取页面内容（包含FrontMatter数据）
    private function getPageContent($page)
    {
        $content = $page->title() . ' ' . $page->value('header.category') . ' ';
        $content .= implode(' ', $page->taxonomy()['tag'] ?? []) . ' ';
        $content .= strip_tags($page->content());
        return mb_strtolower($content);
    }

    // 辅助方法：匹配关键词
    // private function matchKeywords($keywords, $data)
    // {
    //     if (empty($keywords)) return false;

    //     $content = mb_strtolower($data['title'] . ' ' . $data['content']);
    //     foreach ($keywords as $keyword) {
    //         $kw = mb_strtolower(trim($keyword));
    //         if (!empty($kw) && mb_strpos($content, $kw) !== false) {
    //             return true;
    //         }
    //     }
    //     return false;
    // }
    
    private function matchKeywords($keywords, $data)
	{
	    $content = $data['title'] . ' ' . $data['content'];
	    $content = mb_strtolower($content);
	    
	    foreach ($keywords as $keyword) {
	        $kw = mb_strtolower(trim($keyword));
	        if (empty($kw)) continue;
	        
	        // 使用模糊匹配（包含部分词）
	        if (mb_strpos($content, $kw) !== false) {
	            return true;
	        }
	        
	        // 或者使用正则表达式匹配（更灵活）
	        // if (preg_match("/{$kw}/ui", $content)) {
	        //     return true;
	        // }
	    }
	    return false;
	}

    // 辅助方法：生成摘要
    private function getSummary($content, $keywords, $length, $highlight)
    {
        $pureContent = strip_tags($content);
        $pos = 0;
        
        // 查找第一个关键词出现的位置
        foreach ($keywords as $keyword) {
            $kw = mb_strtolower(trim($keyword));
            if (!empty($kw)) {
                $pos = mb_strpos($pureContent, $kw);
                if ($pos !== false && $pos > 50) break;
            }
        }

        $start = max(0, $pos - 20);
        $summary = mb_substr($pureContent, $start, $length);
        $summary = $this->highlight($summary, $keywords, $highlight);
        
        return trim($summary) . (mb_strlen($pureContent) > $length ? '...' : '');
    }

    // 辅助方法：高亮关键词
    private function highlight($text, $keywords, $enable)
    {
        if (!$enable) return $text;

        foreach ($keywords as $keyword) {
            $kw = trim($keyword);
            if (!empty($kw)) {
                $text = preg_replace("/(".$kw.")/i", '<mark>$1</mark>', $text);
            }
        }
        return $text;
    }
}

?>