<?php

namespace Grav\Plugin\PopularArticles;

use Grav\Common\Grav;
use Grav\Plugin\Admin\Popularity;
use Grav\Common\Page\Interfaces\PageInterface;

class PopularArticles
{
    private $popularityFile;
    private $grav;
    private $articles;
    private $config;
    private $popularityData = null;

    public function __construct(array $config)
    {
        $this->grav = Grav::instance();
        $this->config = $config;
        $this->popularityFile = $this->grav['locator']->findResource('log://popularity', true, true)
            . DIRECTORY_SEPARATOR
            . Popularity::TOTALS_FILE;
    }

    public function get()
    {
        if (!$this->articles) {
            $this->articles = $this->build();
        }
        return $this->articles;
    }
    
    /**
     * 根据路由获取阅读数量
     */
    public function getViewsByRoute($route)
    {
        $this->loadPopularityData();
        
        if (isset($this->popularityData[$route])) {
            return $this->popularityData[$route];
        }
        
        return 0;
    }
    
    /**
     * 加载并缓存阅读量数据
     */
    private function loadPopularityData()
    {
        if ($this->popularityData === null) {
            if (file_exists($this->popularityFile)) {
                $this->popularityData = (array)json_decode(
                    file_get_contents($this->popularityFile),
                    true
                );
                
                // 调试日志：加载数据总量
                $this->grav['log']->info("PopularArticles: Loaded popularity data with " . count($this->popularityData) . " routes");
            } else {
                $this->popularityData = [];
                $this->grav['log']->warning("PopularArticles: Popularity file not found: " . $this->popularityFile);
            }
        }
    }

    /**
     * 检查页面是否应被排除
     */
    private function isExcluded(PageInterface $page, string $route, array $excludedRoutes, array $excludedTaxonomies, array $excludedContentTypes): bool
    {
        // 1. 检查路由排除
        foreach ($excludedRoutes as $excludedRoute) {
            if (strpos($route, $excludedRoute) === 0) {
                $this->grav['log']->info("PopularArticles: Excluded by route - $route (matches $excludedRoute)");
                return true;
            }
        }
        
        // 2. 检查分类排除 - 增强支持所有分类类型
        if (!empty($excludedTaxonomies)) {
            $taxonomy = $page->taxonomy() ?? [];
            
            // 检查所有分类类型
            foreach ($taxonomy as $taxType => $taxValues) {
                // 跳过空分类
                if (empty($taxValues)) continue;
                
                // 如果是数组类型的分类值
                if (is_array($taxValues)) {
                    foreach ($excludedTaxonomies as $excludedTax) {
                        if (in_array($excludedTax, $taxValues, true)) {
                            $this->grav['log']->info("PopularArticles: Excluded by taxonomy - $route ($taxType: $excludedTax)");
                            return true;
                        }
                    }
                }
                // 如果是字符串类型的分类值
                elseif (is_string($taxValues)) {
                    foreach ($excludedTaxonomies as $excludedTax) {
                        if ($taxValues === $excludedTax) {
                            $this->grav['log']->info("PopularArticles: Excluded by taxonomy - $route ($taxType: $excludedTax)");
                            return true;
                        }
                    }
                }
            }
        }
        
        // 3. 检查内容类型排除
        if (in_array($page->template(), $excludedContentTypes)) {
            $this->grav['log']->info("PopularArticles: Excluded by content type - $route (type: {$page->template()})");
            return true;
        }
        
        return false;
    }

    public function build()
    {
        $articlesCount = $this->config['articles_count'] ?? 5;
        $blogRoute = $this->config['blog_route'] ?? '/';
        
        // 处理排除配置
        $excludedRoutes = array_map('trim', explode(',', $this->config['excluded_routes'] ?? ''));
        $excludedRoutes = array_filter($excludedRoutes);  // 移除空项
        
        $excludedTaxonomies = array_map('trim', explode(',', $this->config['excluded_taxonomies'] ?? ''));
        $excludedTaxonomies = array_filter($excludedTaxonomies);
        
        $excludedContentTypes = $this->config['excluded_content_types'] ?? [];
        if (!is_array($excludedContentTypes)) {
            $excludedContentTypes = [$excludedContentTypes];
        }

        // 确保加载数据
        $this->loadPopularityData();
        $popularityData = $this->popularityData;
        
        // 调试日志：输出配置
        $this->grav['log']->info("PopularArticles: Config - BlogRoute: $blogRoute, Count: $articlesCount");
        $this->grav['log']->info("PopularArticles: Excluded Routes: " . implode(', ', $excludedRoutes));
        $this->grav['log']->info("PopularArticles: Excluded Taxonomies: " . implode(', ', $excludedTaxonomies));
        $this->grav['log']->info("PopularArticles: Excluded Content Types: " . implode(', ', $excludedContentTypes));
        
        // 过滤数据
        $filteredData = [];
        foreach ($popularityData as $route => $count) {
            // 检查路由是否属于博客部分
            $isBlogRoute = strpos($route, $blogRoute) === 0;
            
            // 检查页面是否存在
            $page = $this->grav['page']->find($route);
            
            if (!$page) {
                $this->grav['log']->info("PopularArticles: Page not found - $route");
                continue;
            }
            
            // 检查是否应被排除
            if ($this->isExcluded($page, $route, $excludedRoutes, $excludedTaxonomies, $excludedContentTypes)) {
                continue;
            }
            
            if ($isBlogRoute) {
                $filteredData[$route] = $count;
            }
        }
        
        // 按访问量降序排序
        arsort($filteredData);
        
        // 限制返回数量（保留键名）
        $filteredData = array_slice($filteredData, 0, $articlesCount, true);
        
        // 创建页面集合
        $pages = [];
        foreach ($filteredData as $route => $count) {
            $page = $this->grav['page']->find($route);
            if ($page) {
                $pages[] = [
                    'page' => $page,
                    'views' => $count,
                    'route' => $route
                ];
            }
        }
        
        // 调试日志：最终结果
        $this->grav['log']->info("PopularArticles: Final articles count: " . count($pages));
        
        return $pages;
    }
}